classdef TA_System <matlab.mixin.Copyable
    % Wet thermoacoustic system
    
    properties
        %system mean pressure [Pa]
        P_m (1,1)double{mustBeLessThan(P_m,1e7)...
            ,mustBeGreaterThan(P_m,9e4)}=1e5 
        
        %system frequncy [Hz]
        Frequency (1,1)double{mustBeLessThan(Frequency,1e6),...
            mustBeGreaterThan(Frequency,1)}=100 
        
        Inert      ... %string with inert component
            (1,:) {mustBeA(Inert,["cell","char"])} ='Air'     
        Reactive   ... %string with reactive component
            (1,:) {mustBeA(Reactive,["cell","char"])}='Water'    
        Dry_Switch (1,1) logical    %1 for dry, 0 for wet
        
        %oscillating pressure ,oscillating velocity and temperature before 
        %the first component. dimensions are [pa] [m/s] [k].
        Begin (1,3) double
    end
    
    properties (SetAccess=?TA_Component)
        %these are the parameters protected from user access
        
        Guesses (1,:)cell              % a vector of guesses
        Targets (1,:)cell              % a vector of targets
        X               (:,1) double   %location [m]
        Pressure        (:,1) double   %pressure amlitude [pa]
        Velocity        (:,1) double   %velocity amplitude [m/s]
        Temperature     ...            %temperature [k]
            (:,1) double{mustBeNonnegative}
        
        Acoustic_Power  (:,1) double   %the acoustic power []
        Total_Power     (:,1) double   %the total power []
        Mass_Flux       (:,1) double   %the mass flux []
        Name            (1,:) char     %string containing name
        Components      (1,:) cell     % a list of component names
        
        %assisting parameters for  Guesses and targets, indicating real
        %part,imaginary part, absolute value or phase
        Guesses_RIAP    (1,:) cell
        Targets_RIAP    (1,:) cell
    end
    
    properties (Hidden)
        Components_H (1,:) cell %handles to actual components
        
        %assisting parameters for  Guesses and targets, indicating format
        Guesses_ID (1,:) cell  
        Targets_ID (1,:) cell   
        
        %a cell array with parameters for calculation of mixture properties
        Mixture_Array  cell 
    end
    methods
        %% constructor
        function obj = TA_System(name,p_m,frequency,inert,reactive,dry_switch,begin)
            obj.Name=name;
            obj.P_m = p_m;
            obj.Inert=inert;
            obj.Frequency=frequency;
            obj.Reactive=reactive;
            obj.Dry_Switch=dry_switch;
            obj.Begin=begin;
            obj.Components={};
            obj.Components_H={};
            obj.Guesses={};
            obj.Targets={};
            obj.Guesses_ID={};
            obj.Targets_ID={};
            obj.Guesses_RIAP={};
            obj.Targets_RIAP={};
        end
        %% set methods for inert and reactive components
        function set.Inert(obj,value)
            obj.Inert=value;
            if ~isempty(obj.Reactive)
                obj.Mixture_Array=collect_properties(obj.Inert,obj.Reactive);
            end
        end
        function set.Reactive(obj,value)
            obj.Reactive=value;
            if ~isempty(obj.Reactive)
                obj.Mixture_Array=collect_properties(obj.Inert,obj.Reactive);
            end
        end
        %% add component to name list
        %thi? function is used internaly, shoudln't be used by the user to
        %add a component. a component is added automatically when it is
        %created if the system is selected under "system"
        function addComponent(obj,component)
            obj.Components=[obj.Components,component];
        end
        %% add a Guess
        function addGuess(obj,varargin)
            % this function adds a guess to the system's guess list
            
            % addGuess(property)- input is a string with a property name
            % e.g addGuess('P_m') sets P_m as a guess
            
            % addGuess(property,i) input is a string with property name
            % and an integer index
            % e.g addGuess('Begin',3) sets Begin(3) (temperature at the
            % begining) as a guess.
            
            % addGuess(component,component_property) input is a string with
            % component name and another one with the component property
            % name.
            % e.g addGuess ('Duct1','length')
            % sets the length of component duct1 as a guess
            
            % addGuess(component,component_property,i) input is a string with
            % component name, another one with the component property
            % name, and an index
            
            %if the added  guess is a complex number, the default option
            %would be to add the real part. if you want to add the
            %imaginary part, phase, or absolute value, you need to add a
            %string with the words 'Imag','Phase',or 'Abs' to it.
            % e.g addGuess('Imag',Begin',1) sets imag(Begin(1)) (real part
            % of pressure the begining) as a guess.
            
            
            N=length(obj.Guesses);
            %check if guess is real, imaginary, phase or absolute
            switch varargin{1}
                case 'Real'
                    varargin=varargin(2:end);
                    obj.Guesses_RIAP{N+1}='R';
                case 'Imag'
                    varargin=varargin(2:end);
                    obj.Guesses_RIAP{N+1}='I';
                case 'Phase'
                    varargin=varargin(2:end);
                    obj.Guesses_RIAP{N+1}='P';
                case 'Abs'
                    varargin=varargin(2:end);
                    obj.Guesses_RIAP{N+1}='A';
                otherwise
                    obj.Guesses_RIAP{N+1}='R';
            end
            
            % case 1
            if length(varargin)==1&&ischar(varargin{1})
                obj.Guesses_ID{N+1}=1;
                try
                    obj.(varargin{1});
                catch
                    error(['no property exists with the name ',varargin{1}])
                end
                %case 2
            elseif length(varargin)==2&&ischar(varargin{1})...
                    &&isnumeric(varargin{2})
                try
                    obj.(varargin{1});
                catch
                    error(['no property exists with the name ',varargin{1}])
                end
                obj.Guesses_ID{N+1}=2;
                %case 3
            elseif length(varargin)==2&&ischar(varargin{1})...
                    &&ischar(varargin{2})
                I=obj.findLoc(varargin{1});
                obj.Guesses_ID{N+1}=[3,I];
                try
                    obj.Components_H{I}.(varargin{2});
                catch
                    error(['no property exists with the name ',varargin{2}])
                end
                %case 4
            elseif length(varargin)==3&&ischar(varargin{1})&&...
                    ischar(varargin{2})&&isnumeric(varargin{3})
                I=obj.findLoc(varargin{1});
                obj.Guesses_ID{N+1}=[4,I];
                try
                    obj.Components_H{I}.(varargin{2});
                catch
                    error(['no property exists with the name ',varargin{2}])
                end
            else
                error('wrong format for adding a Guess or target')
            end
            obj.Guesses{N+1}=varargin;
            
        end
        %% add a Target
        function addTarget(obj,varargin)
            % this function adds a guess to the system's target list
            
            % addTarget(property)- input is a string with a property name
            % e.g addTarget('P_m') sets P_m as a target
            
            % addTarget(property,i) input is a string with property name
            % and an integer index
            % e.g addTarget('Begin',3) sets Begin(3) (temperature at the
            % begining) as a target.
            
            % addTarget(component,component_property) input is a string with
            % component name and another one with the component property
            % name.
            % e.g addTarget ('Duct1','length')
            % sets Begin(3) (temperature at the
            % begining) as a target.
            
            %save original list for insertion into list
            N=length(obj.Targets);
            
            %check if target is real, imaginary, phase or absolute
            switch varargin{1}
                case 'Real'
                    varargin=varargin(2:end);
                    obj.Targets_RIAP{N+1}='R';
                case 'Imag'
                    varargin=varargin(2:end);
                    obj.Targets_RIAP{N+1}='I';
                case 'Phase'
                    varargin=varargin(2:end);
                    obj.Targets_RIAP{N+1}='P';
                case 'Abs'
                    varargin=varargin(2:end);
                    obj.Targets_RIAP{N+1}='A';
                otherwise
                    obj.Targets_RIAP{N+1}='R';
            end
            
            % case 1
            if length(varargin)==1&&ischar(varargin{1})
                obj.Targets_ID{N+1}=1;
                try
                    obj.(varargin{1});
                catch
                    error(['no property exists with the name ',varargin{1}])
                end
                %case 2
            elseif length(varargin)==2&&ischar(varargin{1})...
                    &&isnumeric(varargin{2})
                try
                    obj.(varargin{1});
                catch
                    error(['no property exists with the name ',varargin{1}])
                end
                obj.Targets_ID{N+1}=2;
                %case 3
            elseif length(varargin)==2&&ischar(varargin{1})...
                    &&ischar(varargin{2})
                I=obj.findLoc(varargin{1});
                obj.Targets_ID{N+1}=[3,I];
                try
                    obj.Components_H{I}.(varargin{2});
                catch
                    error(['no property exists with the name ',varargin{2}])
                end
                %case 4
            elseif length(varargin)==3&&ischar(varargin{1})&&...
                    ischar(varargin{2})&&isnumeric(varargin{3})
                I=obj.findLoc(varargin{1});
                obj.Targets_ID{N+1}=[4,I];
                try
                    obj.Components_H{I}.(varargin{2});
                catch
                    error(['no property exists with the name ',varargin{2}])
                end
            else
                error('wrong format for adding a Guess or target')
            end
            obj.Targets{N+1}=varargin;
            
        end
        %% run system without targets or guesses
        function  runSystem(obj,collectdata)
            %this function runs a thermoacoustic system
            %collectdata determines whether data is collected from the system,
            
            %empty all variables
            obj.Pressure=[];
            obj.Velocity=[];
            obj.Temperature=[];
            obj.X=[];
            obj.Mass_Flux=[];
            obj.Total_Power=[];
            obj.Acoustic_Power=[];
            
            if ~collectdata
                %run without collecting data
                myLoc=0;
                %Pressure, Velocity, Temperature
                PUT=obj.Begin;
                %total power
                H=0.5*real(PUT(1)*conj(PUT(2)));
                for i=1:length(obj.Components)
                    %run component
                    [PUT,H]=obj.Components_H{i}.run_component(PUT,H,myLoc);
                    %update location
                    myLoc=myLoc+obj.Components_H{i}.Length;
                end
                
            else
                %run and collect data
                myLoc=0;
                %Pressure, Velocity, Temperature
                PUT=obj.Begin;
                %total power
                H=0.5*real(PUT(1)*conj(PUT(2)));
                for i=1:length(obj.Components)
                    %run component
                    [PUT,H]=obj.Components_H{i}.run_component(PUT,H,myLoc);
                    %update location
                    myLoc=myLoc+obj.Components_H{i}.Length;
                    %calculate derived parameters
                    obj.Components_H{i}.Calculate_Derived
                    %collect data
                    obj.Pressure=[obj.Pressure;obj.Components_H{i}.Pressure];
                    obj.Velocity=[obj.Velocity;obj.Components_H{i}.Velocity];
                    obj.Temperature=[obj.Temperature;obj.Components_H{i}.Temperature];
                    obj.X=[obj.X;obj.Components_H{i}.X];
                    obj.Mass_Flux=[obj.Mass_Flux;obj.Components_H{i}.Mass_Flux];
                    obj.Total_Power=[obj.Total_Power;obj.Components_H{i}.Total_Power];
                    obj.Acoustic_Power=[obj.Acoustic_Power;obj.Components_H{i}.Acoustic_Power];
                end
            end
        end
        %% run system with guesses and targets
        function [X,Fnorm, Nfev,Info, errorInf]=Run_System_GT(obj,guessvalues,targetvalues,varargin)
            %this function runs the system with guesses and targets
            
            %[X,Fvec, Nfev,  Info]=obj.Run_System_GT(guessvalues,targetvalues)
            %[X,Fvec, Nfev,  Info]=obj.Run_System_GT(guessvalues,targetvalues,'Smart')
            %[X,Fvec, Nfev,  Info]=obj.Run_System_GT(guessvalues,targetvalues,'Not_Smart')
            
            %input parameters- guessvalues, targetvalues self explanatory
            %                  if 'Smart' is input, system will attempt several
            %                  ways of converging in case of divergence.
            %                  if  'Not_Smart' no other convergance ways will
            %                  be attempted
            %                  %if neither is input, system will prompt user to
            %                  decide.
            
            if length(obj.Guesses)~=length(obj.Targets)||length(obj.Guesses)...
                    ~=length(guessvalues)||length(guessvalues)~=length(targetvalues)
                error('system must have the same number of guesses, targets and guess values')
            end
            
            %run DNSQ
            func=@(g)(obj.G2T(g,targetvalues));
            [Fvec, Nfev, X, Info,errorInf] = DNSQ(func,guessvalues);
            if ~isempty(errorInf)
                switch errorInf.I
                    case 'Mixture_properties:aboveB'
                        warning(['system went above boiling temperature at'...
                            ,'least once during this run, guess values were [',...
                            num2str(errorInf.X), ']. consider changing ',...
                            'the initial guess. to investigate the error',...
                            ' message, run the system with these values.'])
                    otherwise
                         warning(['system encountered an error at'...
                            ,'least once during this run, guess values were [',...
                            num2str(errorInf.X), ']. consider changing ',...
                            'the initial guess. to investigate the error',...
                            ' message, run the system with these values.'])
                end
            end
            
            
            if Info==0
                fprintf('error, check input parameters')
            elseif Info>1
                if isempty(varargin)
                    %check if should contiunue smartly or not
                    answer = questdlg(['Algorithm did not converge. would ', ...
                        'you like algorithm to attempt fixing it?'], ...
                        'Divergence', 'Yes','No','Yes');
                    % Handle response
                    switch answer
                        case 'Yes'
                            smart=true;
                        case 'No'
                            smart=false;
                    end
                elseif strcmp(varargin{1},'Smart')
                    smart=true;
                elseif strcmp(varargin{1},'Not_Smart')
                    smart=false;
                else
                    error('wrong input format')
                end
                if smart %smart attempts
                    % try rerunning a few times
                    Fnorm=norm(Fvec,2);
                    if Fnorm<2
                        for i=1:3
                            [Fvec, Nfev, X, Info] = DNSQ(func,X);
                            if Info==1
                                break
                            end
                        end
                    end
                    %try relaxing Xtol
                    Fnorm=norm(Fvec,2);
                    if Fnorm<0.1&&~Info~=1
                        [Fvec, Nfev, X, Info] = DNSQ(func,X,1e-4,100,250,1e-12);
                    end
                    %try some random attempts around the original guess
                    if Info~=1
                        for i=1:20
                            Xtry=guessvalues.*(rand(1,length(X))*0.1+0.95);
                            [Fvec, Nfev, X, Info] = DNSQ(func,Xtry);
                            if Info==1
                                break
                            end
                        end
                    end
                    
                end
            end
            %print results and insturcitons
            Fnorm=norm(Fvec,2);
            if Info==1
                fprintf('algorithm converged!\n ')
                fprintf(strrep(['The converged value for guesses is: (' sprintf(' %d,', X) ')'], ',)', ')\n'))
                obj.G2T(X,targetvalues);
                obj.runSystem(1);
                fprintf('number of function evaluations is %d\n',Nfev)
                fprintf('norm of the residuals is %d\n',Fnorm)
            else
                fprintf('targets did not converge. try changing guess \n')
            end
        end
        %% run system with fsolve instead of DNSQ
        function X=Run_System_GT2(obj,guessvalues,targetvalues,varargin)
             if length(obj.Guesses)~=length(obj.Targets)||length(obj.Guesses)...
                    ~=length(guessvalues)||length(guessvalues)~=length(targetvalues)
                error('system must have the same number of guesses, targets and guess values')
            end
            
            %run fsolve
            func=@(g)(obj.G2T(g,targetvalues));
            X = fsolve(func,guessvalues);
        end
        %%  assisting function for running system with targets and guesses
        function target= G2T(obj,guessvalue,targetvalue)
            %this function gets as an input a guessed value of all guessed
            %variables, runs the system and returns the target value
            
            %collect guesses
            for i=1:length(obj.Guesses)
                switch obj.Guesses_ID{i}(1)
                    %different types of guesses, see in "addGuess"
                    case 1
                        new=obj.changeguess(obj.(obj.Guesses{i}{1}),guessvalue(i),obj.Guesses_RIAP{i});
                        obj.(obj.Guesses{i}{1})=new;
                    case 2
                        new=obj.changeguess(obj.(obj.Guesses{i}{1})(obj.Guesses{i}...
                            {2}),guessvalue(i),obj.Guesses_RIAP{i});
                        obj.(obj.Guesses{i}{1})(obj.Guesses{i}{2})=new;
                    case 3
                        new=obj.changeguess(obj.Components_H{obj.Guesses_ID{i}(2)}...
                            .(obj.Guesses{i}{2}),guessvalue(i),obj.Guesses_RIAP{i});
                        obj.Components_H{obj.Guesses_ID{i}(2)}.(obj.Guesses{i}{2})=new;
                    case 4
                        new=obj.changeguess(obj.Components_H{obj.Guesses_ID{i}(2)}...
                            .(obj.Guesses{i}{2})(obj.Guesses{i}{3}),guessvalue(i)...
                            ,obj.Guesses_RIAP{i});
                        obj.Components_H{obj.Guesses_ID{i}(2)}.(obj.Guesses{i}...
                            {2})(obj.Guesses{i}{3})=new;
                end
            end
            
            %run system
            obj.runSystem(0)
            
            %collect targets
            target=zeros(size(guessvalue));
            for i=1:length(obj.Targets)
                switch obj.Targets_ID{i}(1)
                    %different types of guesses, see in "addGuess"
                    case 1
                        target(i)=obj.(obj.Targets{i}{1});
                    case 2
                        target(i)=obj.(obj.Targets{i}{1})(obj.Targets{i}{2});
                    case 3
                        target(i)=obj.Components_H{obj.Targets_ID{i}(2)}.(obj.Targets{i}{2});
                    case 4
                        target(i)=obj.Components_H{obj.Targets_ID{i}(2)}.(obj.Targets{i}{2})(obj.Targets{i}{3});
                end
                switch obj.Targets_RIAP{i}(1)
                    case 'R'
                        %real value
                        if targetvalue(i)==0
                            target(i)=real(target(i));
                        else
                            target(i)=real(target(i))/targetvalue(i)-1;
                            
                        end
                    case 'I'
                        %imaginary value
                        if targetvalue(i)==0
                            target(i)=imag(target(i));
                        else
                            target(i)=imag(target(i))/targetvalue(i)-1;
                        end
                    case 'A'
                        %absolute value
                        if targetvalue(i)==0
                            target(i)=abs(target(i));
                        else
                            target(i)=abs(target(i))/targetvalue(i)-1;
                        end
                    case 'P'
                        %phase
                        if targetvalue(i)==0
                            target(i)=angle(target(i));
                        else
                            target(i)=angle(target(i))/targetvalue(i)-1;
                        end
                end
            end
            
        end
         %% plotting function
        function varargout=PlotSystem(obj,varargin)
            %this function plots the oscilating pressure, oscilating velocity
            %and mean temperature in all the system components
            
            %preperatory work
            if  any(strcmp(varargin,'forGUI'))
                GUI=true;
            else
                GUI=false;
            end
            fs=cell(1,6);
            as=cell(1,6);
            for i=1:6
                try
                    close(i)
                catch
                end
                fs{i}=figure(i);
                as{i}=axes(fs{i});
                hold on
                if GUI
                    fs{i}.Visible='Off'; 
                end
            end
            set(0,'DefaultFigureWindowStyle','docked')
            legends=cell(1,length(obj.Components));
            legends_complex=cell(2,length(obj.Components));             
            if  isempty(obj.Pressure)
                error('system must be run before it is plotted')
            end
            targetcounter=0; %counts number of targets in plot
            % set some properties for all plots
            set(0, 'DefaultAxesFontWeight', 'bold', ...
                'DefaultAxesFontSize', 14, ...
                'DefaultAxesFontWeight', 'bold', ... 
                'DefaultAxesTitleFontWeight', 'bold', ...
                'DefaultAxesTitleFontSizeMultiplier', 2) ;
            set(0, 'DefaultLineLineWidth', 4);
            
            for i=1:length(obj.Components) %run over all components
                if ~isa(obj.Components_H{i},'TA_rget')
                    %plot pressure
                    if any(strcmp(varargin,'Scaled'))
                        yyaxis(as{1},'left')
                        plot1a=plot(as{1},obj.Components_H{i}.X,real(obj.Components_H{i}.Pressure));
                        yyaxis(as{1},'right')
                        plot1b=plot(as{1},obj.Components_H{i}.X,imag(obj.Components_H{i}.Pressure));
                        %make sure both plots have the same style
                        set(plot1b,'LineStyle',get(plot1a,'LineStyle'))
                    else
                        plot1a=plot(as{1},obj.Components_H{i}.X,real(obj.Components_H{i}.Pressure),'-');
                        plot1b=plot(as{1},obj.Components_H{i}.X,imag(obj.Components_H{i}.Pressure),'-..');
                        %make sure both plots have the same color
                        set(plot1b,'color',get(plot1a,'color'))
                    end
                    %plot velocity
                    if any(strcmp(varargin,'Scaled'))
                        yyaxis(as{2},'left')
                        plot2a=plot(as{2},obj.Components_H{i}.X,real(obj.Components_H{i}.Velocity));
                        yyaxis(as{2},'right')
                        plot2b=plot(as{2},obj.Components_H{i}.X,imag(obj.Components_H{i}.Velocity));
                        %make sure both plots have the same style
                        set(plot2b,'LineStyle',get(plot2a,'LineStyle'))
                    else
                        plot2a=plot(as{2},obj.Components_H{i}.X,real(obj.Components_H{i}.Velocity),'-');
                        plot2b=plot(as{2},obj.Components_H{i}.X,imag(obj.Components_H{i}.Velocity),'-..');
                        %make sure both plots have the same color
                        set(plot2b,'color',get(plot2a,'color'))
                    end
                    %plot temperature
                    plot3=plot(as{3},obj.Components_H{i}.X,obj.Components_H{i}.Temperature);
                    if ~any(strcmp(varargin,'Scaled'))
                    %make sure all plots are of the same color
                    set(plot3,'color',get(plot1a,'color'))
                    end
                    %create legends array
                    name1=strrep(obj.Components_H{i}.Name,'_',' ');
                    name2=strcat(name1,' Real');
                    name3=strcat(name1,' imag');
                    legends{i}=name1;
                    legends_complex(:,i)={name2,name3};
                    if any(strcmp(varargin,'Scaled'))
                        %if the plot is scaled, plot handles need to be
                        %stored for legned
                        if i==1
                            legends_list=[];
                        end
                        legends_list=[legends_list,[plot1a;plot1b;plot2a;plot2b]];
                    end
                    %create a plot of derived functions as well
                    if any(strcmp(varargin,'Derived'))
                        plot4=plot(as{4},obj.Components_H{i}.X,obj.Components_H{i}.Acoustic_Power);
                        %make sure all plots are of the same color
                        set(plot4,'color',get(plot3,'color'))
                        plot5=plot(as{5},obj.Components_H{i}.X,obj.Components_H{i}.Total_Power);
                        %make sure all plots are of the same color
                        set(plot5,'color',get(plot3,'color'))
                        plot6=plot(as{6},obj.Components_H{i}.X,obj.Components_H{i}.Mass_Flux);
                        %make sure all plots are of the same color
                        set(plot6,'color',get(plot3,'color'))
                    end
                else %if it is a target
                    legends(i)={nan};
                    legends_complex(:,i)={nan;nan};
                    targetcounter=targetcounter+1;
                end
            end
            % remove nans from legen array
            legends(cellfun(@(legends) any(isnan(legends)),legends)) = [];
            legends_complex(cellfun(@(legends_complex) any(isnan(legends_complex)),legends_complex)) = [];
            %labeling
            as{1}.Title.String='Pressure Distribution';
            as{1}.XLabel.String='x [m]';
            if any(strcmp(varargin,'Scaled'))
                as{1}.YLabel.String='Im P [pa]';
                yyaxis(as{1},'left')
                as{1}.YLabel.String='Re P [pa]';
                list=legends_list(1:2,:);
                list=list(:);
                legend(as{1},list(:),legends_complex)
            else
                as{1}.YLabel.String='P [pa]';
                legend(as{1},legends_complex(:))
            end            
            as{2}.Title.String='Velocity Distribution';
            as{2}.XLabel.String='x [m]';
            if any(strcmp(varargin,'Scaled'))
                as{2}.YLabel.String='Im U [m^3/s]';
                yyaxis(as{2},'left')
                as{2}.YLabel.String='Re U [m^3/s]';
                list=legends_list(3:4,:);
                legend(as{2},list(:),legends_complex)
            else
                as{2}.YLabel.String='U [m^3/s]';    
                legend(as{2},legends_complex(:))
            end
            as{3}.Title.String='Temperature Distribution';
            legend(as{3},legends)
            as{3}.XLabel.String='x [m]';
            as{3}.YLabel.String=('T [k]');
            if any(strcmp(varargin,'Derived'))
                legend(as{4},legends)
                as{4}.XLabel.String='x [m]';
                ylabel(as{4},'$\dot{E}\quad[W] $','Interpreter','latex')
                as{4}.Title.String='Acoustic Power Distribution';
                legend(as{5},legends)
                as{5}.XLabel.String='x [m]';
                ylabel(as{5},'$\dot{H}\quad[W] $','Interpreter','latex')
                as{5}.Title.String='total Power Distribution';
                legend(as{6},legends)
                as{6}.XLabel.String='x [m]';
                ylabel(as{6},'$\dot{m}\quad[kg/s] $','Interpreter','latex')
                as{6}.Title.String='Mass Flux Distribution';
            end
            if GUI
            varargout={fs};
            end
        end
        %% copy system
        function cp=CopySystem(obj,newName,varargin)
            %this function copies a system, creating a new  identical
            %system with copies of all components. the name of the new
            %system will be defined by "new name".
            cp = obj.copyElement;
            cp.Components_H={};
            cp.Name=newName;
            if isempty(varargin)
                clipboard=CopyGroup(obj,[1,length(obj.Components)]);
            else
                clipboard=CopyGroup(obj,[1,length(obj.Components)],varargin{1});
            end
            cp.Components_H=clipboard;
            for i=1:length(clipboard)
                cp.Components_H{i}.System_H=cp;
                cp.Components_H{i}.Location=i;
                cp.Components_H{i}.System=cp.Name;
                cp.Components_H{i}.rename(cp.Components_H{i}.Name);
            end
            if ~isempty(cp.Pressure) %if the original system was already run
                cp.runSystem(1);
            end
        end
        %% insert a component to a system
        % this function inserts a component into the system.
        % input is a handle to the component
        % Insert(component) puts the component at the end of the system
        % Insert(component,Loc) inserts the component into the location
        % specified by "loc"
        
        function Insert(obj,varargin)
            
            % asses and asign input
            if length(varargin)==2
                component=varargin{1};
                newLoc=varargin{2};
            elseif length(varargin)==1
                component=varargin{1};
                newLoc=length(obj.Components)+1;
            else
                error('wrong input format')
            end
            if ~isempty(component.System)
                error('component already has a system')
            end
            if newLoc>length(obj.Components)+1
                error('system has only %d components',length(obj.Components))
            end
            if newLoc<1||floor(newLoc)~=newLoc
                error('location must be a positve integer')
            end
            
            %move components forward
            for i=length(obj.Components)+1:-1:newLoc+1
                obj.Components{i}=obj.Components{i-1};
                obj.Components_H{i}=obj.Components_H{i-1};
                obj.Components_H{i}.Location=i;
            end
            
            %add references between system and object
            obj.Components{newLoc}=component.Name;
            obj.Components_H{newLoc}=component;
            component.System=obj.Name;
            component.System_H=obj;
            component.Location=newLoc;
            obj.Recalc_GT;
        end
        %% remove a component from a system
        function Remove(obj,component)
            %this function removes a component from the system. the
            %input can be either a handle to the component or a string with
            %component name
            
            %find location of object in the system and convert from name to
            %actual component if neccesary
            if ischar(component)
                Loc=obj.findLoc(component);
                component=obj.Components_H{Loc};
            else
                Loc=obj.findLoc(component.Name);
            end
            
            %check if object has a guess or a target associated with it
            [~,isGuess]=obj.SearchArray(obj.Guesses,component.Name,[]);
            [~,isTarget]=obj.SearchArray(obj.Targets,component.Name,[]);
            if isGuess
                error(['remove component "',component.Name,'" from guess list before removing it from system',...
                    'use the Function "Remove_Guess"'])
            end
            if isTarget
                error(['remove component "',component.Name,'" from target list before removing it from system',...
                    'use the Function "Remove_Target"'])
            end
            
            %remove system references from component
            component.System='';
            component.System_H=TA_System.empty;
            component.Location=[];
            
            %rearange system component to disclude components
            for i=Loc:length(obj.Components)-1
                obj.Components{i}=obj.Components{i+1};
                obj.Components_H{i}=obj.Components_H{i+1};
                obj.Components_H{i}.Location=i;
            end
            obj.Components_H(end)=[];
            obj.Components(end)=[];
            
            %redefine guesses and targets
            obj.Recalc_GT;
            
        end
        %% replace a system component
        function Replace(obj,old,new)
            %this function removes a component from the system. the
            %input can be either a handle to the component or a string with
            %component name
            
            %find location of object in the system and convert from name to
            %actual component if neccesary
            if ischar(old)
                Loc=obj.findLoc(old);
                old=obj.Components_H{Loc};
            else
                Loc=obj.findLoc(old.Name);
            end
            
            %check if object has a guess or a target associated with it
            [~,isGuess]=obj.SearchArray(obj.Guesses,old.Name,[]);
            [~,isTarget]=obj.SearchArray(obj.Targets,old.Name,[]);
            if isGuess
                error(['remove component "',old.Name,'" from guess list before removing it from system',...
                    'use the Function "Remove_Guess"'])
            end
            if isTarget
                error(['remove component "',old.Name,'" from target list before removing it from system',...
                    'use the Function "Remove_Target"'])
            end
            
            %replace references
            old.System=[];
            old.Location=[];
            old.System_H=[];
            
            new.System=obj.Name;
            new.System_H=obj;
            new.location=Loc;
            
            obj.Components_H(Loc)=new;
            obj.Components(Loc)=new.Name;
        end
        %% remove Guess from system
        %this function removes the nth guess from the guess list
        function Remove_Guess(obj,n)
            if length(obj.Guesses)<n
                error('system has only %d guesses',length(obj.Guesses))
            end
            obj.Guesses(n)=[];
            obj.Guesses_RIAP(n)=[];
            obj.Guesses_ID(n)=[];
        end
        %% remove Target from system
        %this function removes the nth target from the target list
        function Remove_Target(obj,n)
            if length(obj.Targets)<n
                error('system has only %d targets',length(obj.Targets))
            end
            obj.Targets(n)=[];
            obj.Targets_RIAP(n)=[];
            obj.Targets_ID(n)=[];
        end
        
        %% assisting function for finding the location of a component
        function I=findLoc(obj,componentName)
            %this function finds the location of a component in the system
            %input is a string with component name
            I=0;
            for i=1:length(obj.Components)
                if strcmp(obj.Components(i),componentName)
                    I=i;
                end
            end
            if ~I
                error(['no component exists in system with the name ',componentName])
            end
        end
        %% assisting function for reassigning all guesses and targets
        function Recalc_GT(obj)
            
            %collect all guesses and targets
            Oldguess=obj.Guesses;
            Oldtarg=obj.Targets;
            Old_g_RIAP=obj.Guesses_RIAP;
            Old_t_RIAP=obj.Targets_RIAP;
            
            %empty all
            obj.Guesses={};
            obj.Targets={};
            obj.Guesses_ID={};
            obj.Targets_ID={};
            
            for i=1:length(Oldguess)
                obj.addGuess(Oldguess{i}{:})
            end
            for i=1:length(Oldtarg)
                obj.addTarget(Oldtarg{i}{:})
            end
            obj.Guesses_RIAP=Old_g_RIAP;
            obj.Targets_RIAP=Old_t_RIAP;
        end
        %% assisting function for collecting guess values from system
        function values=CollectGuessValues(obj)
            values=zeros(1,length(obj.Guesses));
            for i=1:length(obj.Guesses)
                switch obj.Guesses_ID{i}(1)
                    %different types of guesses, see in "addGuess"
                    case 1
                        values(i)=obj.(obj.Guesses{i}{1});
                    case 2
                        values(i)=obj.(obj.Guesses{i}{1})(obj.Guesses{i}{2});
                    case 3
                        values(i)=obj.Components_H{obj.Guesses_ID{i}(2)}.(obj.Guesses{i}{2});
                    case 4
                        values(i)= obj.Components_H{obj.Guesses_ID{i}(2)}.(obj.Guesses{i}...
                            {2})(obj.Guesses{i}{3});
                end
                switch obj.Guesses_RIAP{i}
                    case 'R'
                        values(i)=real(values(i));
                    case 'I'
                        values(i)=imag(values(i));
                    case 'A'
                        values(i)=abs(values(i));
                    case 'P'
                        values(i)=angle(values(i));
                end  
                
            end
        end
        %% assisting function for collecting target values from system
        function values=CollectTargetValues(obj)
            values=zeros(1,length(obj.Targets));
            for i=1:length(obj.Targets)
                switch obj.Targets_ID{i}(1)
                    %different types of guesses, see in "addTarget"
                    case 1
                        values(i)=obj.(obj.Targets{i}{1});
                    case 2
                        values(i)=obj.(obj.Targets{i}{1})(obj.Targets{i}{2});
                    case 3
                        values(i)=obj.Components_H{obj.Targets_ID{i}(2)}.(obj.Targets{i}{2});
                    case 4
                        values(i)= obj.Components_H{obj.Targets_ID{i}(2)}.(obj.Targets{i}...
                            {2})(obj.Targets{i}{3});
                end
                switch obj.Targets_RIAP{i}
                    case 'R'
                        values(i)=real(values(i));
                    case 'I'
                        values(i)=imag(values(i));
                    case 'A'
                        values(i)=abs(values(i));
                    case 'P'
                        values(i)=angle(values(i));
                end
                
            end
            
        end
        %% Assisting function for changing system property
        % this function changes a system property as specified by an
        % array
        
        % Change_Property({property},value)- input is a string with a property name
        % e.g Change_Property({'P_m'}) changes P_m.
        
        % Change_Property({property,i},value) input is a string with property name
        % and an integer index
        % e.g Change_Property('Begin',3) changes Begin(3) (temperature at the
        % begining)
        
        % Change_Property({component,component_property},value) input is a string with
        % component name and another one with the component property
        % name.
        % e.g Change_Property ({'Duct1','length'},value)
        % changes the length of component Duct1
        
        % Change_Property({component,component_property},value) input is a string with
        % component name, another one with the component property
        % name, and an index.
        % e.g Change_Property ({'Duct1','Pressure',1},value) changes the pressure of
        % at the begining of Duct1
        
        function Change_Property(obj,info_array,value)
            % case 1
            if length(info_array)==1&&ischar(info_array{1})
                obj.(info_array{1})=value;
                %case 2
            elseif length(info_array)==2&&ischar(info_array{1})...
                    &&isnumeric(info_array{2})
                obj.(info_array{1})(info_array{2})=value;
                %case 3
            elseif length(info_array)==2&&ischar(info_array{1})...
                    &&ischar(info_array{2})
                I=obj.findLoc(info_array{1});
                obj.Components_H{I}.(info_array{2})=value;
                %case 4
            elseif length(info_array)==3&&ischar(info_array{1})&&...
                    ischar(info_array{2})&&isnumeric(info_array{3})
                I=obj.findLoc(info_array{1});
                obj.Components_H{I}.(info_array{2})(info_array{3})=value;
            else
                error('wrong format for changing a property')
            end
        end
        %% Assisting function for evaluating system property
        % this function evaluates a system property as specified by an
        % array
        
        % Evaluate_Property({property})- input is a string with a property name
        % e.g Evaluate_Property({'P_m'}) evaluates P_m.
        
        % Evaluate_Property({property,i}) input is a string with property name
        % and an integer index
        % e.g Evaluate_Property('Begin',3) evaluates Begin(3) (temperature at the
        % begining)
        
        % Evaluate_Property({component,component_property}) input is a string with
        % component name and another one with the component property
        % name.
        % e.g Evaluate_Property ({'Duct1','length'})
        % evaluates the length of component Duct1
        
        % Evaluate_Property(component,component_property,i) input is a string with
        % component name, another one with the component property
        % name, and an index.
        % e.g Evaluate_Property ({'Duct1','Pressure',1}) evaluates the pressure of
        % at the begining of Duct1
        
        function value=Evaluate_Property(obj,info_array)
            % case 1
            if length(info_array)==1&&ischar(info_array{1})
                value=obj.(info_array{1});
                try
                    obj.(info_array{1});
                catch
                    error(['no property exists with the name ',info_array{1}])
                end
                %case 2
            elseif length(info_array)==2&&ischar(info_array{1})...
                    &&isnumeric(info_array{2})
                value=obj.(info_array{1})(info_array{2});
                %case 3
            elseif length(info_array)==2&&ischar(info_array{1})...
                    &&ischar(info_array{2})
                I=obj.findLoc(info_array{1});
                value=obj.Components_H{I}.(info_array{2});
                %case 4
            elseif length(info_array)==3&&ischar(info_array{1})&&...
                    ischar(info_array{2})&&isnumeric(info_array{3})
                I=obj.findLoc(info_array{1});
                value=obj.Components_H{I}.(info_array{2})(info_array{3});
            else
                error('wrong format for evaluating a property')
            end
        end
    end
    methods(Static)
        %% assisting function for changing a guesed value
        function  new=changeguess(old,value,RIAP)
            switch RIAP
                case 'R'
                    %real value
                    new=value+1i*imag(old);
                case 'I'
                    %imaginary value
                    new=1i*value+real(old);
                case 'A'
                    %absolute value
                    [a1,a2]=pol2cart(angle(old),value);
                    new=a1+1i*a2;
                case 'P'
                    %phase
                    [a1,a2]=pol2cart(value,abs(old));
                    new=a1+1i*a2;
            end
        end
        %% assisting function for checking and manipulating the guess and targets array
        function [I_Array,value]=SearchArray(I_Array,Old,New)
            %this function searches the array defined by I_Array based on the
            % Expression in "Old". if New is empty it will simply return true
            % if it found the expression and false otherwise. if new is a string it will
            % replace the old string with the new
            %usually a guess or target array.
            %if C=1 - the function
            value=false;
            if isempty(New)
                flag=0;
            else
                flag=1;
            end
            for i=1:length(I_Array)
                if strcmp(I_Array{i}{1},Old)
                    value=true;
                    if flag
                        I_Array{i}{1}=New;
                    else
                        return
                    end
                end
            end
        end
        %% Assisting function for comparing one array with an array of other arrays
        function answer=Comparearray(A1,A2)
            %this function compares the small array A1 with the large
            %array A2, comparing A1 to every value of A2. returns the
            %location of A1 in A2 or 0 if no such location exists.
            fun=@(a,b) length(a)==length(b)&&max(cellfun(@isequal, {a},{b}));
            answer=find(cellfun(@(c) fun(A1,c),A2));
            if isempty(answer)
                answer=0;
            end
        end
    end
    
end

